/**
   JPath
   by Josh Nimoy
   for Gabe Dunne
   23 jan 2007
*/

#include "JPath.h"
#include "ofMain.h"
//#include "jttoolkit.h"
//#include "Global.h"
#include "GeometryUtils.h"
#include <fstream>

using namespace std;

/**
 point on curve by jtnimoy and Jeremy Rotsztain
 */

JPoint JPath::pointOnCurve(float gradient){
	
	JPoint retVal;
	
	switch(points.size()){
		case 0: break;
		case 1: retVal.copyFrom(points[0]); break;
		case 2: retVal.copyFrom(points[0].lerpTo(points[1],gradient)); break;
		default:
			float targetDistance = gradient * getLength();
			//accumilate distance
			float accum = 0;
			deque<JPoint>::iterator it;
			bool foundOne = false;
			for(it = (points.begin() + 1);it!=points.end();it++){
				float dist = (it-1)->distanceFrom(*it);
				if(accum + dist >= targetDistance){
					//we reached one segment beyond the end so go back and do the fraction.
					retVal = (it-1)->lerpTo( *it , (targetDistance - accum) / dist);
					foundOne = true;
					break;
				}else accum += dist;
			}
			
			if(!foundOne){//got to the end of the for loop?
				retVal.copyFrom(points.back());//if we got here, it's 100%
			}
			break;

	}

	return retVal;
}

//-------------------------------------------------------------------------------

void JPath::updateFrame(int frameNum){
	if(frameNum>=frameCount)frameNum = frameCount-1;
	for(int i=0;i<vertexCount;i++){
		points[i].copyFrom(animation[frameNum*vertexCount+i]);
	}
}


//-------------------------------------------------------------------------------

void JPath::loadDat(const char*filename){
	ifstream infile(filename,ios::binary);
	infile.read((char*)&frameCount,4);
	infile.read((char*)&vertexCount,4);
	
	float tangents_bool;
	float normals_bool;
	infile.read((char*)&tangents_bool,4);
	infile.read((char*)&normals_bool,4);
	
	for(int i=0;i<frameCount*vertexCount;i++){
		
		if(!infile.good())valid = false;
		
		JPoint p;
		infile.read((char*)&(p.x),4);
		infile.read((char*)&(p.y),4);
		infile.read((char*)&(p.z),4);
		
		//fill the initial slots as well
		if(i<vertexCount){
			points.push_back(p);
		}
		
		animation.push_back(p);
		
		if(tangents_bool){
			infile.read((char*)&(p.x),4);
			infile.read((char*)&(p.y),4);
			infile.read((char*)&(p.z),4);
			animation_tangents.push_back(p);
		}
		
		if(normals_bool){
			infile.read((char*)&(p.x),4);
			infile.read((char*)&(p.y),4);
			infile.read((char*)&(p.z),4);
			animation_normals.push_back(p);
		}
	}
}


void interpolateBezier(
					   float P0X,float P0Y,float P0Z,
					   float P1X,float P1Y,float P1Z,
					   float P2X,float P2Y,float P2Z,
					   float P3X,float P3Y,float P3Z,
					   int segs,
					   float Xs[] , float Ys[] , float Zs[]) {

    float t=0.0f,dt=1.0f/(float)(segs-1);		// calculating increment
    float Q0X,Q0Y,Q0Z;
    float Q1X,Q1Y,Q1Z;
    float Q2X,Q2Y,Q2Z;
    float R0X,R0Y,R0Z;
    float R1X,R1Y,R1Z;
    float x;float y;float z;
    float xx=P0X;float yy=P0Y;float zz=P0Z;
    for (int i=0;i<segs;i++) {
		Q0X = P0X + t*(P1X-P0X); Q0Y = P0Y + t*(P1Y-P0Y); Q0Z = P0Z + t*(P1Z-P0Z);
		Q1X = P1X + t*(P2X-P1X); Q1Y = P1Y + t*(P2Y-P1Y); Q1Z = P1Z + t*(P2Z-P1Z);
		Q2X = P2X + t*(P3X-P2X); Q2Y = P2Y + t*(P3Y-P2Y); Q2Z = P2Z + t*(P3Z-P2Z);
		R0X = Q0X + t*(Q1X-Q0X); R0Y = Q0Y + t*(Q1Y-Q0Y); R0Z = Q0Z + t*(Q1Z-Q0Z);
		R1X = Q1X + t*(Q2X-Q1X); R1Y = Q1Y + t*(Q2Y-Q1Y); R1Z = Q1Z + t*(Q2Z-Q1Z);
		x = R0X + t*(R1X-R0X);
		y = R0Y + t*(R1Y-R0Y);
		z = R0Z + t*(R1Z-R0Z);
		//drawLine(xx,yy,zz,x,y,z);
		Xs[i]=x;
		Ys[i]=y;
		Zs[i]=z;
		xx=x;
		yy=y;
		zz=z;
		t+=dt;
    }
}



JPath::JPath(JPath *mentor){
	JPath(mentor->points);
}


JPath::JPath(){
  resetVars();
}

JPath::JPath(deque<JPoint>p){
  resetVars();

  //make a deep copy of that vector
  deque<JPoint>::iterator it;
  for(it=p.begin();it!=p.end();it++){
    points.push_back(JPoint(it->x,it->y,it->z));
  }
}

JPath::JPath(vector<JPoint>p){
  resetVars();

  //make a deep copy of that vector
  vector<JPoint>::iterator it;
  for(it=p.begin();it!=p.end();it++){
    points.push_back(JPoint(*it));
  }
}



void JPath::resetVars(){
  stroke_width = 1.5;
  useInnerFill = true;
  useOutline = false;
  stroke_flare = 10;
  stroke_style = 0;
  valid = true;
}


void JPath::fromBezier(JPoint p1,JPoint p2,JPoint p3,JPoint p4,int steps){

  // allocate some working space.
  float *Xs = new float[steps];
  float *Ys = new float[steps];
  float *Zs = new float[steps];

  interpolateBezier(
		    p1.x,p1.y,p1.z,
		    p2.x,p2.y,p2.z,
		    p3.x,p3.y,p3.z,
		    p4.x,p4.y,p4.z,
		    steps,Xs,Ys,Zs) ;


  points.push_back(JPoint(p1));
  for(int i=1;i<steps-1;i++){
    points.push_back(JPoint(Xs[i],Ys[i],Zs[i]));
  }
  points.push_back(JPoint(p4));


  delete Xs;
  delete Ys;
  delete Zs;

}




JPath::~JPath(){  
/*
 deque<JPoint*>::iterator it;
  for(it=points.begin();it!=points.end();it++){
    delete (*it);
  }
 */
}

/**
   divide all edges in 2.
*/
void JPath::subdiv(){
  deque<JPoint>dest;


  deque<JPoint>::iterator it;
  int i=0;
  for(it=points.begin();it!=points.end();it++){
    dest.push_back(*it);
    if(i<points.size()-1){//don't add one for the last
      dest.push_back(JPoint( it->lerpTo(*(it+1),0.5)    ));
    }
    i++;
  }





  //ok, now copy dest back into points
  points.clear();
  for(it=dest.begin();it!=dest.end();it++){
    points.push_back(*it);
  }

}


/**
 loop through all the verts and call glVertex3f(x,y,z);
 */
void JPath::glVertex(){
	deque<JPoint>::iterator it;
	for(it=points.begin();it!=points.end();it++){
		it->glVertex();
	}
	
}




/**
 loop through all the verts and call glVertex3f(x,y,z);
 */
void JPath::ofVert(){
	deque<JPoint>::iterator it;
	for(it=points.begin();it!=points.end();it++){
		ofVertex(it->x,it->y);
	}
	
}



/*
  lerping towards the previous and next points, we are effectively
  melting the geometry toward the center.
*/
void JPath::melt(float heat){
  //draw handles
  deque<JPoint>::iterator it;
  int i=0;
  for(it=points.begin();it!=points.end();it++){

    deque<JPoint>::iterator curr = it;
    deque<JPoint>::iterator prev;
    deque<JPoint>::iterator next;

    if(i>0)prev=it-1;
    else   prev=curr;

    if(i<points.size()-1)next=it+1;
    else   next=curr;

    curr->lerpSelfTo(*prev,heat);
    curr->lerpSelfTo(*next,heat);

    i++;//keep this close to the end curly bracket.
  }
}

void JPath::smooth(int val){
  for(int i=0;i<val;i++){
    subdiv();
    melt(0.5);
  }
}

void JPath::println(){

  cout << "JPath( ";
  deque<JPoint>::iterator it;
  for(it=points.begin();it!=points.end();it++){
    JPoint t = *it;
    cout << "JPoint(" << t.x << ' ' << t.y << ' ' << t.z << ") ";
  }
  cout << " ) " << endl;
}

void JPath::render(){
	deque<JPoint>::iterator it;
	glBegin(GL_LINE_STRIP);
	for(it=points.begin();it!=points.end();it++){
		it->glVertex();
	}
	glEnd();
}

void JPath::renderPoints(){
	deque<JPoint>::iterator it;
	glBegin(GL_POINTS);
	for(it=points.begin();it!=points.end();it++){
		it->glVertex();
	}
	glEnd();
}

void JPath::reset(){
	points.clear();
}

void JPath::renderAsBezier(int steps){
  JPath p;
  p.fromBezier(points[0],points[1],points[2],points[3],steps);
  p.render();
}





void JPath::stroke(int steps,bool bez=false){
	if(points.size()<2)return;
	deque<JPoint>::iterator it;
	JPath *p;
	if(bez){
		p = new JPath();
		p->fromBezier(points[0],points[1],points[2],points[3],steps);
	}else p = new JPath(points);
	
	deque<JPoint>row1;
	deque<JPoint>row2;
	deque<JPoint>row3;
	
	int i=0;
	for(it=p->points.begin();it!=p->points.end();it++){
		JPoint p1;
		JPoint p2;
		JPoint p3;
		JPoint p4;
		
		if(i==0){//beginning case
			p1.copyFrom(*it);
			p2.copyFrom(*(it+1));
			p3.copyFrom(p2);
			p4.copyFrom(p2);
			p3.rotateSelfZ( PI/2,p1);
			p4.rotateSelfZ(-PI/2,p1);
		}else if(i==points.size()-1){//end case
			p1.copyFrom(*(it-1));
			p2.copyFrom(*(it));
			p3.copyFrom(p1);
			p4.copyFrom(p1);
			p3.rotateSelfZ(-PI/2,p2);
			p4.rotateSelfZ( PI/2,p2);
		}else{//in the middle, we average the angles.
			p1.copyFrom(*(it-1));
			p2.copyFrom(*(it));
			p3.copyFrom(p1);
			p4.copyFrom(p1);
			p3.rotateSelfZ(-PI/2,p2);
			p4.rotateSelfZ( PI/2,p2);
			
			//calculate averagables
			JPoint p5(p1);
			JPoint p6(p1);
			p5.rotateSelfZ(-PI/2,p2);
			p6.rotateSelfZ( PI/2,p2);
			p3.lerpSelfTo(p5,0.5);
			p4.lerpSelfTo(p6,0.5);
		}
		//accumilate the results for later.
		row1.push_back(JPoint(p3));
		row2.push_front(JPoint(p4));
		row3.push_back(JPoint(*it));
		
		////normalise the outstretches away from the skel.
		//row1.back().normalizeSelfFrom(row3.back(),10);
		//row2.front().normalizeSelfFrom(row3.back(),10);
		i++;
	}
	delete p;
	
	//MASSAGE THE SHAPE
	for(int i=0;i<row1.size();i++){
		int ii = row1.size()-1-i;
		
		
		//  >>===---===<<
		//float fatness1 = stroke_width*2-pow((stroke_width*(float)i)/row1.size() * (stroke_width*(float)ii)/row1.size(),2);
		float fatness1 = stroke_width*stroke_flare -
		(stroke_width* ((float)i/row1.size() * (stroke_width*(float)ii)/row1.size()));
		
		
		
		//   ---==+++==--
		float fatness2 = (stroke_width*(float)i)/row1.size() * (stroke_width*(float)ii)/row1.size();
		float fatness3 = row1[i].distanceFrom( row2[row2.size()-1-i-1]);
		
		
		float fatness;
		if(stroke_style==0){
			fatness = fatness1*points.size();
		}else if(stroke_style==1){
			fatness = fatness3*points.size()*0.001;
		}else{
			float p =points.size()*0.1;
			fatness =  fmax(1,(stroke_width*100 - fmin(stroke_width*100 ,fatness3*2)))*0.5;
		}
		
		
		row1[ i].normalizeSelfFrom(row3[i],fatness);
		row2[ii].normalizeSelfFrom(row3[i],fatness);
		
		//magnets to buckle down the edges
	}
	
	//draw buff
	
	if(useInnerFill){
		glBegin(GL_TRIANGLE_STRIP);
		for(int i=0;i<row2.size();i++){
			row2[row2.size()-1-i].glVertex();
			row1[i].glVertex();
		}
		glEnd();
	}
	
	
	if(useOutline){
		glBegin(GL_LINE_LOOP);
		for(int i=0;i<row1.size();i++)row1[i].glVertex();
		for(int i=0;i<row2.size();i++)row2[i].glVertex();
		glEnd();
	}
	
/*	
	 //draw ladder spokes.
	 glBegin(GL_LINES);
	 for(int i=0;i<row1.size();i++){
	 int ii = row1.size()-1-i;
	 row1[ i].glVertex();
	 row2[ii].glVertex();
	 }
	 glEnd();
	
	//draw middle skeleton
	
	 glBegin(GL_LINE_STRIP);
	 for(int i=0;i<row3.size();i++){
	 row3[i].glVertex();
	 }
	 glEnd();
*/	 

	
}








void JPath::drink(FingerPoints f, int offset){
  deque<JPoint>::iterator it;
  int i=0;
  for(it=points.begin();it!=points.end();it++){
    (*it)+=f[offset+i];
    i++;
  }
}



JPoint JPath::average(){

 JPoint accum;
  deque<JPoint>::iterator it;
  for(it=points.begin();it!=points.end();it++){
    accum+=(*it);
  }
  return accum/(float)points.size();

  return 0;
}


void JPath::shrink(float v){

  JPoint accum;
  JPoint a = average();
  deque<JPoint>::iterator it;
  for(it=points.begin();it!=points.end();it++){
    it->lerpSelfTo(a,v);
  }

}

void JPath::operator+=(JPoint p){
	
	deque<JPoint>::iterator it;
	for(it=points.begin();it!=points.end();it++){
		(*it)+=p;
	}
	 
}

void JPath::operator*=(JPoint p){
	
	deque<JPoint>::iterator it;
	for(it=points.begin();it!=points.end();it++){
		(*it)*=p;
	}
	 
}

void JPath::operator/=(JPoint p){
	
	deque<JPoint>::iterator it;
	for(it=points.begin();it!=points.end();it++){
		(*it)/=p;
	}
	 
}


void JPath::operator+=(float f){
	
	deque<JPoint>::iterator it;
	for(it=points.begin();it!=points.end();it++){
		(*it)+=f;
	}
	
}

void JPath::operator*=(float f){
	
	deque<JPoint>::iterator it;
	for(it=points.begin();it!=points.end();it++){
		(*it)*=f;
	}
	
}

void JPath::operator/=(float f){
	
	deque<JPoint>::iterator it;
	for(it=points.begin();it!=points.end();it++){
		(*it)/=f;
	}
	
}




void JPath::eatPointsFromBothEnds(int bites){
	
  for(int i=0;i<bites;i++){

    if(points.size()>0){
      points.pop_front();
    }

    if(points.size()>0){
      points.pop_back();
    }

  }
	
}

//------------------------------------------------------
void JPath::strokeRandomOverlap(float width,float angle){
	
	if(points.size()<2)return;
	glLineWidth(2);
	//edge by edge
	deque<int*> chunks;
	bool polarity=false;
	int start=0;
	for(int i=1;i<points.size();i++){
		float a = atan2(points[i].y-points[i-1].y,points[i].x-points[i-1].x);
		if(polarity!=a<angle || i==points.size()-1){//draw the line						
			int *f = new int[2];
			f[0]=start;
			f[1]=i;
			if(rnd[chunks.size()]>0.5)chunks.push_back(f);
			else chunks.push_front(f);
			start = i-1;
		}
		polarity = (a<angle);
	}
	
	for(int i=0;i<chunks.size();i++){
		//fill
		glColor4f(0.5,0.5,1,0.5);
		glBegin(GL_TRIANGLE_STRIP);
		for(int j=chunks[i][0];j<chunks[i][1];j++){
			JPoint p(points[j]);
			p+=JPoint((width/2),0,0);
			p.rotateSelfZ(angle,points[j]);
			p.glVertex();
			p.rotateSelfZ(PI,points[j]);
			p.glVertex();
		}				
		glEnd();
		
		glColor4f(1,1,1,0.5);
		
		glBegin(GL_LINE_STRIP);
		
		JPoint firstPoint;
		//side 1 of stroke
		for(int j=chunks[i][0];j<chunks[i][1];j++){
			JPoint p(points[j]);
			p+=JPoint((width/2),0,0);
			p.rotateSelfZ(angle,points[j]);
			p.glVertex();
			if(j==chunks[i][0])firstPoint.copyFrom(p);
		}
		//side 2 of stroke
		for(int j=chunks[i][1];j>=chunks[i][0];j--){
			JPoint p(points[j]);
			p+=JPoint((-width/2),0,0);
			p.rotateSelfZ(angle,points[j]);
			p.glVertex();
		}
		firstPoint.glVertex();
		
		glEnd();
	}
	
	//delete chunks
	deque<int*>::iterator it;
	for(it=chunks.begin();it<chunks.end();it++)delete [] *it;
	 
	
}
//------------------------------------------------------
 void JPath::strokeRibbon(float width,float angle){
	 
	if(points.size()<2)return;

	 for(int i=0;i<3;i++){
		if(i==0)glBegin(GL_TRIANGLE_STRIP);
		else glBegin(GL_LINE_STRIP);
		 for(int j=0;j<points.size();j++){
			JPoint p(points[j]);
			p+=JPoint((width/2),0,0);
			p.rotateSelfZ(angle,points[j]);
			if(i!=1)p.glVertex();
			p.rotateSelfZ(PI,points[j]);
			if(i!=2)p.glVertex();
		}
		glEnd();
	 }
	 
}
//------------------------------------------------------
void JPath::outlineTube(float width){
	glBegin(GL_LINE_STRIP);
	vertexTube(width,false);
	glEnd();
}
//------------------------------------------------------
void JPath::vertexTube(float width, bool ofVert){
	
	if(points.size()<2)return;
	
	JPoint firstPoint;
	//side 1 of stroke
	deque<float>angles;
	for(int j=0;j<points.size();j++){
		JPoint p(points[j]);
		p+=JPoint((width/2),0,0);
		
		float angle;
		if(j==0)angle = atan2(points[j+1].y-points[j].y,points[j+1].x-points[j].x);
		else angle = atan2(points[j].y-points[j-1].y,points[j].x-points[j-1].x);
		angles.push_back(angle);
		
		p.rotateSelfZ(angle-PI/2,points[j]);
		if(ofVert)ofVertex(p.x,p.y);
		else glVertex2f(p.x,p.y);
		if(j==0)firstPoint.copyFrom(p);
	}
	//side 2 of stroke
	for(int j=points.size()-1;j>=0;j--){
		JPoint p(points[j]);
		p+=JPoint((-width/2),0,0);
		p.rotateSelfZ(angles[j]-PI/2,points[j]);
		if(ofVert)ofVertex(p.x,p.y);
		else glVertex2f(p.x,p.y);
	}
	if(ofVert)ofVertex(firstPoint.x,firstPoint.y);
	else glVertex2f(firstPoint.x,firstPoint.y);
}
//---------------------------------------------------
void JPath::strokePaint(float width){
	
	if(points.size()<2)return;

	//line1
	for(int i = 0; i <3;i++){
		if(i==0)glBegin(GL_TRIANGLE_STRIP);//fill
		else glBegin(GL_LINE_STRIP);// 2 outlines
		for(int j=0;j<points.size();j++){
			JPoint p(points[j]);
			float angle;
			float segLength;
			if(j==0){
				angle = atan2(points[j+1].y-points[j].y,points[j+1].x-points[j].x);
				segLength = p.distanceFrom(points[j+1]);
			}else{
				angle = atan2(points[j].y-points[j-1].y,points[j].x-points[j-1].x);
				segLength = p.distanceFrom(points[j-1]);
			}		
			
			//  >>===---===<<
			float fatness = fmax(1,width - pow(segLength,1.2) );
			
			p+=JPoint((fatness/2),0,0);
			p.rotateSelfZ(angle-PI/2,points[j]);
			if(i!=2)p.glVertex();
			p.rotateSelfZ(PI,points[j]);
			if(i!=1)p.glVertex();
		}
		glEnd();
		
		//caps
		if(i==0)ofFill();
		else ofNoFill();
		ofCircle(points.front().x,points.front().y,width/2);
		//ofCircle(points.back()->x,points.back()->y,width/2);
		
	}
	
	
}


//-----------------------------------------------------
JPoint JPath::getTopLeft(){
	
	JPoint ret;
	deque<JPoint>::iterator it;
	for(it=points.begin();it<points.end();it++){
		if(it==points.begin()){
			ret.x = it->x;
			ret.y = it->y;
		}else{
			if(ret.x>it->x)ret.x = it->x;
			if(ret.y>it->y)ret.y = it->y;
		}
	}
	
	return ret;
	
}

//---------------------------------------------------
JPoint JPath::getBottomRight(){
	
	JPoint ret;
	deque<JPoint>::iterator it;
	for(it=points.begin();it<points.end();it++){
		if(it==points.begin()){
			ret.x = it->x;
			ret.y = it->y;
		}else{
			if(ret.x<it->x)ret.x = it->x;
			if(ret.y<it->y)ret.y = it->y;
		}
	}
	
	return ret;
	
}
//---------------------------------------------------
JPoint JPath::getCenter(){
	return getTopLeft() + (getBottomRight()-getTopLeft())/2;
}
//---------------------------------------------------
float JPath::getLength(){
	
	float ret;
	deque<JPoint>::iterator it;
	for(it=points.begin();it<points.end();it++){
		if(it!=points.begin()){
			ret+=it->distanceFrom(*(it-1));
		}
	}
	
	return ret;
	
}

//----------------------------------------------------
//creates an aligned pointer
void JPath::push_back_point(JPoint p){
	points.push_back(JPoint(p));
}


//-----------------------------------------------------

/**
	gets rid of redundant segments
 */

void JPath::simplify(float tolerance){
	if(points.size()<3)return;//dont bother if not enough points.
	
	deque<JPoint> accum;
	for(int i=0;i<points.size();i++){
		if(i==0){
			accum.push_back(points[i]);//always add the first
		}else if(i==points.size()-1){
			accum.push_back(points[i]);//always add the last
		}else{
			//get the angle of this point from the last point and the next.
			float a1 = points[i-1].thetaFrom(points[i]);
			float a2 = points[i].thetaFrom(points[i+1]);
			if(fabs(a1-a2)>tolerance){
				accum.push_back(points[i]);
			}
		}
	}
	points = accum;
}
//--------------------------------------------------------

/**
 adds bevel corners to angles
 */

void JPath::bevelAngles(float bevelAmount){
	if(points.size()<3)return;//dont bother if not enough points.
	
	deque<JPoint> accum;
	for(int i=0;i<points.size();i++){
		if(i==0){
			accum.push_back(points[i]);//always add the first
		}else if(i==points.size()-1){
			accum.push_back(points[i]);//always add the last
		}else{
			//get the angle of this point from the last point and the next.
			float a1 = points[i-1].thetaFrom   (points[i]);
			float d1 = points[i-1].distanceFrom(points[i]);
			
			float a2 = points[i+1].thetaFrom   (points[i]);
			float d2 = points[i+1].distanceFrom(points[i]);
			
			float newD1 = d1-fmin(bevelAmount,d1/2);
			float newD2 = d2-fmin(bevelAmount,d2/2);
						
			float x1 = points[i-1].x + newD1 * cos(a1);
			float y1 = points[i-1].y + newD1 * sin(a1);
			
			float x2 = points[i+1].x + newD2 * cos(a2);
			float y2 = points[i+1].y + newD2 * sin(a2);
			
			accum.push_back(JPoint(x1,y1));
			accum.push_back(JPoint(x2,y2));
		}
	}
	points = accum;
	
}


//-----------------------------------------------------

deque<JPoint> JPath::applyCap(JPoint a,JPoint b, float angleDisplace){
	
	deque<JPoint> ret;
	
	
	float thetaA = a.thetaFrom(b);
	thetaA -= PI/2;
	
	float thetaB = b.thetaFrom(a);
	thetaB -= PI/2;
	
	a.x += angleDisplace * cos(thetaA);
	a.y += angleDisplace * sin(thetaA);
	
	b.x += angleDisplace * cos(thetaB);
	b.y += angleDisplace * sin(thetaB);
	
	//return result
	ret.push_back(a);
	ret.push_back(b);
	return ret;
}

//-----------------------------------------------------



deque<JPoint> JPath::applyJoins(deque<JPoint>orig){
	
	deque<JPoint> ret;
	
	float miter_limit = 75;
	
	if(orig.size()>2){
		for(int j=0;j<orig.size();j+=2){
			if(j==0){ // PATH BEGINNING
				JPoint p3(orig[j  ]);
				JPoint p4(orig[j+1]);		
				JPoint p5(orig[j+2]);
				JPoint p6(orig[j+3]);
				
				float retX,retY;
				GeometryUtils::findLinesIntersection(p3.x,p3.y,p4.x,p4.y,p5.x,p5.y,p6.x,p6.y,retX,retY);
				
				
				ret.push_back(orig[j]);
				
				JPoint newPoint(retX,retY);
				if(!isnan(retX)){
					if(orig[j+1].distanceFrom(newPoint) > miter_limit){
						float theta = orig[j+1].thetaFrom(newPoint);
						float r = orig[j+1].distanceFrom(newPoint);
						
						newPoint.x = orig[j+1].x + miter_limit * cos(theta);
						newPoint.y = orig[j+1].y + miter_limit * sin(theta);
						//newPoint.copyFrom(orig[j+1]);
						
					}					
					ret.push_back(newPoint);
				}else ret.push_back(orig[j+1]);
				
				
			}else if(j==orig.size()-2){ // PATH ENDING
				JPoint p1(orig[j-2]);
				JPoint p2(orig[j-1]);
				JPoint p3(orig[j  ]);
				JPoint p4(orig[j+1]);
				
				float retX,retY;
				GeometryUtils::findLinesIntersection(p1.x,p1.y,p2.x,p2.y,p3.x,p3.y,p4.x,p4.y,retX,retY);
				
				JPoint newPoint(retX,retY);
				if(!isnan(retX)){
					if(orig[j].distanceFrom(newPoint) > miter_limit){
						float theta = orig[j].thetaFrom(newPoint);
						float r = orig[j].distanceFrom(newPoint);
						
						newPoint.x = orig[j].x + miter_limit * cos(theta);
						newPoint.y = orig[j].y + miter_limit * sin(theta);
						//newPoint.copyFrom(orig[j]);
						
					}					
					ret.push_back(newPoint);
				}else ret.push_back(orig[j]);
				
				
				
				ret.push_back(orig[j+1]);
				
			}else{ //PATH MIDDLE
				JPoint p1(orig[j-2]);
				JPoint p2(orig[j-1]);
				JPoint p3(orig[j  ]);
				JPoint p4(orig[j+1]);
				JPoint p5(orig[j+2]);
				JPoint p6(orig[j+3]);
				
				float retX,retY;
				float retX2,retY2;
				GeometryUtils::findLinesIntersection(p1.x,p1.y,p2.x,p2.y,p3.x,p3.y,p4.x,p4.y,retX,retY);
				GeometryUtils::findLinesIntersection(p3.x,p3.y,p4.x,p4.y,p5.x,p5.y,p6.x,p6.y,retX2,retY2);
				
				
				JPoint newPoint(retX,retY);
				if(!isnan(retX)){
					if(orig[j].distanceFrom(newPoint) > miter_limit){
						float theta = orig[j].thetaFrom(newPoint);
						float r = orig[j].distanceFrom(newPoint);
						
						newPoint.x = orig[j].x + miter_limit * cos(theta);
						newPoint.y = orig[j].y + miter_limit * sin(theta);
						//newPoint.copyFrom(orig[j]);
					}					
					ret.push_back(newPoint);
				}else ret.push_back(orig[j]);
				
				
				JPoint newPoint2(retX2,retY2);
				if(!isnan(retX2)){
					if(orig[j+1].distanceFrom(newPoint2) > miter_limit){
						float theta = orig[j+1].thetaFrom(newPoint2);
						float r = orig[j+1].distanceFrom(newPoint2);
						
						newPoint2.x = orig[j+1].x + miter_limit * cos(theta);
						newPoint2.y = orig[j+1].y + miter_limit * sin(theta);
						//newPoint.copyFrom(orig[j+1]);
					}					
					ret.push_back(newPoint2);
				}else ret.push_back(orig[j+1]);
				
				
				
			}
		}
	}else{
		return orig;
	}
	return ret;
}


//-----------------------------------------------------

void JPath::strokeJoinCap(float width, float capAngle1,float capAngle2){
	
	if(points.size()<2)return;
	
	deque<JPoint> side1;
	deque<JPoint> side2;
	
	//calculate raw sides
	for(int j=0;j<points.size()-1;j++){
		//side 1
		JPoint p1(points[j]);
		JPoint p2(points[j+1]);
		p1+=JPoint((width/2),0,0);
		p2+=JPoint((width/2),0,0);
		
		float angle = atan2(points[j+1].y - points[j].y , points[j+1].x - points[j].x);
		
		p1.rotateSelfZ(angle-PI/2,points[j]);
		p2.rotateSelfZ(angle-PI/2,points[j+1]);
		side1.push_back(p1);
		side1.push_back(p2);
		
		//side 2
		p1.copyFrom(points[points.size()-j-1]);
		p2.copyFrom(points[points.size()-j-2]);
		p1+=JPoint((width/2),0,0);
		p2+=JPoint((width/2),0,0);
		
		angle = atan2(points[points.size()-j-2].y - points[points.size()-j-1].y , points[points.size()-j-2].x - points[points.size()-j-1].x);
		
		p1.rotateSelfZ(angle-PI/2,points[points.size()-j-1]);
		p2.rotateSelfZ(angle-PI/2,points[points.size()-j-2]);
		side2.push_back(p1);
		side2.push_back(p2);
		
	}
	
	//apply miter joins
	deque<JPoint> side1_joined = applyJoins(side1);
	deque<JPoint> side2_joined = applyJoins(side2);
	
	deque<JPoint> cap1 = applyCap(side1_joined.back(),side2_joined.front(),capAngle1);
	deque<JPoint> cap2 = applyCap(side2_joined.back(),side1_joined.front(),capAngle2);
	
	
	/*
	 //draw the center line
	 glColor3f(0.5,0.5,0.5);
	 glBegin(GL_LINE_STRIP);
	 for(int i=0;i<points.size();i++){
	 points[i].glVertex();
	 }
	 glEnd();
	 */
	
	
	//draw the outer line
	glColor4f(1,1,1,1);
	
	ofBeginShape();
	
	for(int i=0;i<side1_joined.size();i++){
		if(i%2==0){
			ofVertex(side1_joined[i].x,side1_joined[i].y);
		}
	}	
	
	//first cap
	ofVertex(cap1[0].x,cap1[0].y);
	ofVertex(cap1[1].x,cap1[1].y);
	
	for(int i=0;i<side2_joined.size();i++){
		if(i%2==0){
			ofVertex(side2_joined[i].x,side2_joined[i].y);
		}
	}
	
	//second cap
	ofVertex(cap2[0].x,cap2[0].y);
	ofVertex(cap2[1].x,cap2[1].y);
	
	ofEndShape(true);	
	
	
	glBegin(GL_LINE_LOOP);
	
	for(int i=0;i<side1_joined.size();i++){
		if(i%2==1 && i!=side1_joined.size()-1)glVertex2f(side1_joined[i].x,side1_joined[i].y);
	}	
	
	//first cap
	glVertex2f(cap1[0].x,cap1[0].y);
	glVertex2f(cap1[1].x,cap1[1].y);
	
	for(int i=0;i<side2_joined.size();i++){
		if(i%2==0 && i!=0)glVertex2f(side2_joined[i].x,side2_joined[i].y);
	}
	
	//second cap
	glVertex2f(cap2[0].x,cap2[0].y);
	glVertex2f(cap2[1].x,cap2[1].y);
	
	glEnd();
	
}


deque< deque<JPoint> > JPath::getTriangleStrip(float width){

	if(points.size()<2)return deque<deque<JPoint> >();
	
	deque<JPoint> side1;
	deque<JPoint> side2;
	
	//calculate raw sides
	for(int j=0;j<points.size()-1;j++){
		//side 1
		JPoint p1(points[j]);
		JPoint p2(points[j+1]);
		p1+=JPoint((width/2),0,0);
		p2+=JPoint((width/2),0,0);
		
		float angle = atan2(points[j+1].y - points[j].y , points[j+1].x - points[j].x);
		
		p1.rotateSelfZ(angle-PI/2,points[j]);
		p2.rotateSelfZ(angle-PI/2,points[j+1]);
		side1.push_back(JPoint(p1));
		side1.push_back(JPoint(p2));
		
		p1.rotateSelfZ(PI,points[j]);
		p2.rotateSelfZ(PI,points[j+1]);
		side2.push_back(p1);
		side2.push_back(p2);
		
	}
	
	//draw the outer line
	
	deque< deque<JPoint> > ret;
	ret.push_back(side1);
	ret.push_back(side2);
	return ret;
}

/**
 tries to break the path into more even segments.
 */
void JPath::segment(float segLength){
	deque<JPoint> newPoints;
	if(points.size()<2)return;
	deque<JPoint>::iterator iter;
	for(iter=points.begin();iter!=points.end();iter++){
		if(iter==points.begin())newPoints.push_back(*iter);
		else{
			//measure the distance
			float dist = (iter-1)->distanceFrom(*iter);
			if(dist>segLength){
				int segCount = ceil(dist / segLength);
				for(int j=0;j<segCount;j++){
					JPoint p = (iter-1)->lerpTo(*iter,(j+1)/(float)segCount);
					newPoints.push_back(p);
				}
			}else{
				newPoints.push_back(*iter);
			}
		}
	}
	points = newPoints;
}


void JPath::operator<<(JPoint p){
	points.push_back(p);
}

